// 全局错误处理工具
export class GlobalErrorHandler {
  private static instance: GlobalErrorHandler;
  private errorCount = 0;
  private maxErrors = 10;

  private constructor() {
    this.setupGlobalErrorHandlers();
  }

  public static getInstance(): GlobalErrorHandler {
    if (!GlobalErrorHandler.instance) {
      GlobalErrorHandler.instance = new GlobalErrorHandler();
    }
    return GlobalErrorHandler.instance;
  }

  private setupGlobalErrorHandlers() {
    // 捕获未处理的 Promise 错误
    window.addEventListener('unhandledrejection', (event) => {
      this.handleError(event.reason);
      event.preventDefault();
    });

    // 捕获全局 JavaScript 错误
    window.addEventListener('error', (event) => {
      this.handleError(event.error || event.message);
      event.preventDefault();
    });

    // 重写 console.error 来捕获更多错误信息
    const originalConsoleError = console.error;
    console.error = (...args) => {
      this.handleConsoleError(args);
      originalConsoleError.apply(console, args);
    };
  }

  private handleError(error: any) {
    if (this.errorCount >= this.maxErrors) {
      return; // 防止错误循环
    }

    const errorMessage = error?.message || String(error);
    
    // 忽略特定的 DOM 相关错误
    if (this.shouldIgnoreError(errorMessage)) {
      console.warn('忽略的错误:', errorMessage);
      return;
    }

    this.errorCount++;
    console.error('全局错误处理器捕获:', error);
  }

  private handleConsoleError(args: any[]) {
    const errorMessage = args.join(' ');
    
    if (this.shouldIgnoreError(errorMessage)) {
      return;
    }

    this.errorCount++;
  }

  private shouldIgnoreError(errorMessage: string): boolean {
    const ignoredErrors = [
      'getBoundingClientRect',
      'Cannot read properties of null',
      'Cannot read property \'getBoundingClientRect\' of null',
      'Script error',
      'ResizeObserver loop limit exceeded',
      'Non-Error promise rejection captured'
    ];

    return ignoredErrors.some(ignored => 
      errorMessage.toLowerCase().includes(ignored.toLowerCase())
    );
  }

  public resetErrorCount() {
    this.errorCount = 0;
  }
}

// 初始化全局错误处理器
export const initializeErrorHandler = () => {
  GlobalErrorHandler.getInstance();
};