import React from 'react';
import { motion, AnimatePresence } from 'framer-motion';
import { Button } from './ui/button';
import { Badge } from './ui/badge';
import { 
  LayoutDashboard, 
  Users, 
  Calendar, 
  Vote,
  UserCheck,
  Zap,
  ChevronLeft,
  ChevronRight,
  Sparkles,
  Dices,
  GraduationCap,
  Settings
} from 'lucide-react';

interface SidebarProps {
  activeSection: string;
  onSectionChange: (section: string) => void;
  collapsed: boolean;
  onToggleCollapse: () => void;
}

const navigationItems = [
  { 
    id: 'dashboard', 
    label: '仪表盘', 
    icon: LayoutDashboard, 
    color: 'text-blue-600', 
    bgColor: 'bg-blue-50',
    description: '系统概览和快速操作'
  },
  { 
    id: 'students', 
    label: '学生管理', 
    icon: Users, 
    color: 'text-green-600', 
    bgColor: 'bg-green-50',
    description: '管理学生名单和信息'
  },
  { 
    id: 'picker', 
    label: '智能点名', 
    icon: Dices, 
    color: 'text-purple-600', 
    bgColor: 'bg-purple-50',
    description: '随机选择学生',
    badge: 'AI'
  },
  { 
    id: 'attendance', 
    label: '考勤管理', 
    icon: UserCheck, 
    color: 'text-teal-600', 
    bgColor: 'bg-teal-50',
    description: '学生出勤统计'
  },
  { 
    id: 'classroom', 
    label: '课堂工具', 
    icon: GraduationCap, 
    color: 'text-orange-600', 
    bgColor: 'bg-orange-50',
    description: '分组、排座等功能'
  },
  { 
    id: 'schedule', 
    label: '课程安排', 
    icon: Calendar, 
    color: 'text-indigo-600', 
    bgColor: 'bg-indigo-50',
    description: '课程时间管理'
  },
  { 
    id: 'voting', 
    label: '课堂投票', 
    icon: Vote, 
    color: 'text-pink-600', 
    bgColor: 'bg-pink-50',
    description: '实时投票功能'
  },
  { 
    id: 'advanced', 
    label: '高级功能', 
    icon: Zap, 
    color: 'text-yellow-600', 
    bgColor: 'bg-yellow-50',
    description: '权重设置、历史记录'
  },
  { 
    id: 'settings', 
    label: '系统设置', 
    icon: Settings, 
    color: 'text-slate-600', 
    bgColor: 'bg-slate-50',
    description: '个性化配置'
  }
];

export function Sidebar({ activeSection, onSectionChange, collapsed, onToggleCollapse }: SidebarProps) {
  return (
    <motion.div 
      className="fixed left-0 top-0 h-screen bg-white/95 backdrop-blur-xl border-r border-gray-200/50 shadow-xl z-50"
      initial={false}
      animate={{ width: collapsed ? 64 : 288 }}
      transition={{ duration: 0.3, ease: "easeInOut" }}
    >
      {/* 头部 */}
      <div className="flex items-center justify-between p-4 border-b border-gray-200/50">
        <AnimatePresence mode="wait">
          {!collapsed && (
            <motion.div
              initial={{ opacity: 0, x: -20 }}
              animate={{ opacity: 1, x: 0 }}
              exit={{ opacity: 0, x: -20 }}
              transition={{ duration: 0.2 }}
              className="flex items-center space-x-3"
            >
              <div className="w-10 h-10 bg-gradient-to-br from-blue-500 to-purple-600 rounded-xl flex items-center justify-center shadow-lg">
                <Sparkles className="w-6 h-6 text-white" />
              </div>
              <div>
                <h1 className="font-bold text-lg text-gray-900">柯猿课堂互动系统</h1>
                <p className="text-xs text-gray-500">Keyuan Class Connect</p>
              </div>
            </motion.div>
          )}
        </AnimatePresence>
        
        <Button
          variant="ghost"
          size="sm"
          onClick={onToggleCollapse}
          className="p-2 hover:bg-gray-100 rounded-lg transition-colors"
        >
          {collapsed ? (
            <ChevronRight className="w-4 h-4 text-gray-600" />
          ) : (
            <ChevronLeft className="w-4 h-4 text-gray-600" />
          )}
        </Button>
      </div>

      {/* 导航菜单 */}
      <nav className="flex-1 p-3 space-y-1 overflow-y-auto">
        {navigationItems.map((item, index) => {
          const Icon = item.icon;
          const isActive = activeSection === item.id;
          
          return (
            <motion.div
              key={item.id}
              initial={{ opacity: 0, x: -20 }}
              animate={{ opacity: 1, x: 0 }}
              transition={{ delay: index * 0.05 }}
            >
              <Button
                variant="ghost"
                className={`w-full justify-start h-12 rounded-xl transition-all duration-200 group ${
                  isActive 
                    ? 'bg-gradient-to-r from-blue-500 to-purple-600 text-white shadow-lg hover:shadow-xl' 
                    : `text-gray-700 hover:${item.bgColor} hover:${item.color} hover:shadow-sm`
                } ${collapsed ? 'px-3' : 'px-4'}`}
                onClick={() => onSectionChange(item.id)}
                title={collapsed ? item.label : undefined}
              >
                <div className={`flex items-center ${collapsed ? 'justify-center' : 'justify-start'} w-full`}>
                  <div className={`p-2 rounded-lg ${isActive ? 'bg-white/20' : item.bgColor} transition-colors group-hover:scale-110`}>
                    <Icon className={`w-5 h-5 ${isActive ? 'text-white' : item.color} transition-colors`} />
                  </div>
                  
                  <AnimatePresence mode="wait">
                    {!collapsed && (
                      <motion.div
                        initial={{ opacity: 0, x: -10 }}
                        animate={{ opacity: 1, x: 0 }}
                        exit={{ opacity: 0, x: -10 }}
                        transition={{ duration: 0.2 }}
                        className="flex items-center justify-between w-full ml-3"
                      >
                        <div className="flex flex-col items-start">
                          <span className="font-medium text-sm">{item.label}</span>
                          <span className={`text-xs ${isActive ? 'text-white/80' : 'text-gray-500'}`}>
                            {item.description}
                          </span>
                        </div>
                        
                        {item.badge && (
                          <Badge 
                            variant={isActive ? "secondary" : "outline"} 
                            className={`text-xs font-medium ${
                              isActive ? 'bg-white/20 text-white border-white/30' : 'border-gray-300'
                            }`}
                          >
                            {item.badge}
                          </Badge>
                        )}
                      </motion.div>
                    )}
                  </AnimatePresence>
                </div>
              </Button>
            </motion.div>
          );
        })}
      </nav>

      {/* 底部状态 */}
      <AnimatePresence mode="wait">
        {!collapsed && (
          <motion.div
            initial={{ opacity: 0, y: 20 }}
            animate={{ opacity: 1, y: 0 }}
            exit={{ opacity: 0, y: 20 }}
            transition={{ duration: 0.2 }}
            className="p-4 border-t border-gray-200/50"
          >
            <div className="bg-gradient-to-r from-green-50 to-blue-50 rounded-xl p-3">
              <div className="flex items-center justify-between mb-2">
                <span className="text-sm font-medium text-gray-700">系统状态</span>
                <div className="flex items-center space-x-2">
                  <div className="w-2 h-2 bg-green-500 rounded-full animate-pulse"></div>
                  <span className="text-xs text-green-600 font-medium">正常运行</span>
                </div>
              </div>
              <div className="text-xs text-gray-500">
                © 2025 柯猿课堂互动系统 v0.0.2
              </div>
            </div>
          </motion.div>
        )}
      </AnimatePresence>
    </motion.div>
  );
}