import React, { useState, useRef, useEffect } from 'react';
import { motion } from 'framer-motion';
import { Card, CardContent, CardHeader, CardTitle } from './ui/card';
import { Button } from './ui/button';
import { Badge } from './ui/badge';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from './ui/select';
import { Input } from './ui/input';
import { Label } from './ui/label';
import { useToast } from './ui/use-toast';
import { Shuffle, Users, UserCheck, Play, RotateCw, Target } from 'lucide-react';
import type { Student, PickResult, Settings } from '../App';

interface PickerControlProps {
  students: Student[];
  onPickResult: (result: PickResult) => void;
  settings: Settings;
}

type PickMode = 'single' | 'multiple' | 'group';

export function PickerControl({ students, onPickResult, settings }: PickerControlProps) {
  const [mode, setMode] = useState<PickMode>('single');
  const [multipleCount, setMultipleCount] = useState(3);
  const [groupCount, setGroupCount] = useState(2);
  const [isAnimating, setIsAnimating] = useState(false);
  const [showWheel, setShowWheel] = useState(false);
  const [wheelRotation, setWheelRotation] = useState(0);
  const [selectedStudent, setSelectedStudent] = useState<Student | null>(null);
  const canvasRef = useRef<HTMLCanvasElement>(null);
  const { toast } = useToast();

  const playSound = () => {
    if (!settings.soundEnabled) return;
    
    // 创建音频上下文
    const audioContext = new (window.AudioContext || (window as any).webkitAudioContext)();
    const oscillator = audioContext.createOscillator();
    const gainNode = audioContext.createGain();
    
    oscillator.connect(gainNode);
    gainNode.connect(audioContext.destination);
    
    // 根据音效类型设置不同的频率
    switch (settings.soundType) {
      case 'bell':
        oscillator.frequency.setValueAtTime(800, audioContext.currentTime);
        break;
      case 'applause':
        oscillator.frequency.setValueAtTime(400, audioContext.currentTime);
        break;
      case 'music':
        oscillator.frequency.setValueAtTime(600, audioContext.currentTime);
        break;
    }
    
    gainNode.gain.setValueAtTime(settings.volume, audioContext.currentTime);
    gainNode.gain.exponentialRampToValueAtTime(0.01, audioContext.currentTime + 0.5);
    
    oscillator.start(audioContext.currentTime);
    oscillator.stop(audioContext.currentTime + 0.5);
  };

  const shuffleArray = <T,>(array: T[]): T[] => {
    const shuffled = [...array];
    for (let i = shuffled.length - 1; i > 0; i--) {
      const j = Math.floor(Math.random() * (i + 1));
      [shuffled[i], shuffled[j]] = [shuffled[j], shuffled[i]];
    }
    return shuffled;
  };

  // 绘制转盘
  const drawWheel = () => {
    const canvas = canvasRef.current;
    if (!canvas || students.length === 0) return;

    const ctx = canvas.getContext('2d');
    if (!ctx) return;

    const centerX = canvas.width / 2;
    const centerY = canvas.height / 2;
    const radius = Math.min(centerX, centerY) - 20;
    const anglePerSegment = (2 * Math.PI) / students.length;

    // 清空画布
    ctx.clearRect(0, 0, canvas.width, canvas.height);

    // 绘制转盘扇形
    students.forEach((student, index) => {
      const startAngle = index * anglePerSegment + (wheelRotation * Math.PI / 180);
      const endAngle = (index + 1) * anglePerSegment + (wheelRotation * Math.PI / 180);

      // 扇形颜色
      const colors = [
        '#FF6B6B', '#4ECDC4', '#45B7D1', '#96CEB4', '#FFEAA7',
        '#DDA0DD', '#98D8C8', '#F7DC6F', '#BB8FCE', '#85C1E9'
      ];
      ctx.fillStyle = colors[index % colors.length];

      // 绘制扇形
      ctx.beginPath();
      ctx.moveTo(centerX, centerY);
      ctx.arc(centerX, centerY, radius, startAngle, endAngle);
      ctx.closePath();
      ctx.fill();

      // 绘制边框
      ctx.strokeStyle = '#fff';
      ctx.lineWidth = 2;
      ctx.stroke();

      // 绘制文字
      const textAngle = startAngle + anglePerSegment / 2;
      const textX = centerX + Math.cos(textAngle) * (radius * 0.7);
      const textY = centerY + Math.sin(textAngle) * (radius * 0.7);

      ctx.save();
      ctx.translate(textX, textY);
      ctx.rotate(textAngle + Math.PI / 2);
      ctx.fillStyle = '#fff';
      ctx.font = 'bold 14px Arial';
      ctx.textAlign = 'center';
      ctx.fillText(student.name, 0, 0);
      ctx.restore();
    });

    // 绘制中心圆
    ctx.beginPath();
    ctx.arc(centerX, centerY, 30, 0, 2 * Math.PI);
    ctx.fillStyle = '#333';
    ctx.fill();
    ctx.strokeStyle = '#fff';
    ctx.lineWidth = 3;
    ctx.stroke();

    // 绘制指针
    ctx.beginPath();
    ctx.moveTo(centerX, centerY - radius - 10);
    ctx.lineTo(centerX - 15, centerY - radius + 10);
    ctx.lineTo(centerX + 15, centerY - radius + 10);
    ctx.closePath();
    ctx.fillStyle = '#FF4757';
    ctx.fill();
    ctx.strokeStyle = '#fff';
    ctx.lineWidth = 2;
    ctx.stroke();
  };

  // 转盘动画
  const spinWheel = () => {
    if (students.length === 0) return;

    setIsAnimating(true);
    playSound();

    // 随机旋转角度 (3-6圈 + 随机角度)
    const spins = 3 + Math.random() * 3;
    const finalAngle = spins * 360 + Math.random() * 360;
    
    const duration = settings.animationEnabled ? 
      (settings.animationSpeed === 'slow' ? 4000 : 
       settings.animationSpeed === 'normal' ? 3000 : 2000) : 1000;

    const startTime = Date.now();
    const startRotation = wheelRotation;

    const animate = () => {
      const elapsed = Date.now() - startTime;
      const progress = Math.min(elapsed / duration, 1);
      
      // 缓动函数 (ease-out)
      const easeOut = 1 - Math.pow(1 - progress, 3);
      const currentRotation = startRotation + finalAngle * easeOut;
      
      setWheelRotation(currentRotation % 360);

      if (progress < 1) {
        requestAnimationFrame(animate);
      } else {
        // 计算选中的学生
        const normalizedAngle = (360 - (currentRotation % 360) + 90) % 360; // 加90度调整指针位置
        const segmentAngle = 360 / students.length;
        const selectedIndex = Math.floor(normalizedAngle / segmentAngle) % students.length;
        const selected = students[selectedIndex];
        
        setSelectedStudent(selected);
        setIsAnimating(false);

        const result: PickResult = {
          students: [selected],
          timestamp: new Date(),
          mode: 'single'
        };

        onPickResult(result);

        toast({
          title: "转盘停止",
          description: `恭喜 ${selected.name} 被选中！`,
        });
      }
    };

    requestAnimationFrame(animate);
  };

  // 画布大小调整和绘制
  useEffect(() => {
    const canvas = canvasRef.current;
    if (!canvas || !showWheel) return;

    const resizeCanvas = () => {
      const container = canvas.parentElement;
      if (container) {
        const size = Math.min(container.clientWidth, container.clientHeight);
        canvas.width = size;
        canvas.height = size;
        // 延迟绘制确保canvas已经准备好
        setTimeout(() => drawWheel(), 10);
      }
    };

    resizeCanvas();
    window.addEventListener('resize', resizeCanvas);
    return () => window.removeEventListener('resize', resizeCanvas);
  }, [students, wheelRotation, showWheel]);

  // 重绘转盘
  useEffect(() => {
    if (showWheel && canvasRef.current) {
      setTimeout(() => drawWheel(), 10);
    }
  }, [students, wheelRotation, showWheel]);

  const handlePick = async () => {
    if (students.length === 0) {
      toast({
        title: "无法点名",
        description: "请先导入学生名单",
        variant: "destructive",
      });
      return;
    }

    setIsAnimating(true);
    
    // 播放音效
    playSound();

    // 动画延迟
    const animationDelay = settings.animationEnabled ? 
      (settings.animationSpeed === 'slow' ? 2000 : 
       settings.animationSpeed === 'normal' ? 1500 : 1000) : 500;

    setTimeout(() => {
      let pickedStudents: Student[] = [];
      
      switch (mode) {
        case 'single':
          const randomIndex = Math.floor(Math.random() * students.length);
          pickedStudents = [students[randomIndex]];
          break;
          
        case 'multiple':
          const count = Math.min(multipleCount, students.length);
          const shuffled = shuffleArray(students);
          pickedStudents = shuffled.slice(0, count);
          break;
          
        case 'group':
          const groupSize = Math.ceil(students.length / groupCount);
          const shuffledForGroup = shuffleArray(students);
          // 为演示，只返回第一组
          pickedStudents = shuffledForGroup.slice(0, groupSize);
          break;
      }

      const result: PickResult = {
        students: pickedStudents,
        timestamp: new Date(),
        mode
      };

      onPickResult(result);
      setIsAnimating(false);

      toast({
        title: "点名完成",
        description: `已选中 ${pickedStudents.length} 位学生`,
      });
    }, animationDelay);
  };

  const getModeIcon = (currentMode: PickMode) => {
    switch (currentMode) {
      case 'single': return <UserCheck className="h-4 w-4" />;
      case 'multiple': return <Users className="h-4 w-4" />;
      case 'group': return <Shuffle className="h-4 w-4" />;
    }
  };

  const getModeDescription = (currentMode: PickMode) => {
    switch (currentMode) {
      case 'single': return '随机选择一位学生';
      case 'multiple': return `随机选择 ${multipleCount} 位学生`;
      case 'group': return `随机分为 ${groupCount} 组`;
    }
  };

  return (
    <Card className="h-full">
      <CardHeader>
        <CardTitle className="flex items-center space-x-2">
          <Play className="h-5 w-5" />
          <span>点名控制</span>
        </CardTitle>
      </CardHeader>
      
      <CardContent className="space-y-6">
        {/* 点名模式选择 */}
        <div className="space-y-3">
          <Label className="text-sm font-medium">点名模式</Label>
          <Select value={mode} onValueChange={(value: PickMode) => setMode(value)}>
            <SelectTrigger>
              <SelectValue />
            </SelectTrigger>
            <SelectContent>
              <SelectItem value="single">
                <div className="flex items-center space-x-2">
                  <UserCheck className="h-4 w-4" />
                  <span>单人模式</span>
                </div>
              </SelectItem>
              <SelectItem value="multiple">
                <div className="flex items-center space-x-2">
                  <Users className="h-4 w-4" />
                  <span>多人模式</span>
                </div>
              </SelectItem>
              <SelectItem value="group">
                <div className="flex items-center space-x-2">
                  <Shuffle className="h-4 w-4" />
                  <span>分组模式</span>
                </div>
              </SelectItem>
            </SelectContent>
          </Select>
        </div>

        {/* 模式参数设置 */}
        {mode === 'multiple' && (
          <div className="space-y-2">
            <Label htmlFor="multiple-count" className="text-sm font-medium">
              选择人数
            </Label>
            <Input
              id="multiple-count"
              type="number"
              min="1"
              max={students.length}
              value={multipleCount}
              onChange={(e) => setMultipleCount(Math.max(1, parseInt(e.target.value) || 1))}
              className="w-full"
            />
          </div>
        )}

        {mode === 'group' && (
          <div className="space-y-2">
            <Label htmlFor="group-count" className="text-sm font-medium">
              分组数量
            </Label>
            <Input
              id="group-count"
              type="number"
              min="2"
              max={Math.floor(students.length / 2) || 2}
              value={groupCount}
              onChange={(e) => setGroupCount(Math.max(2, parseInt(e.target.value) || 2))}
              className="w-full"
            />
          </div>
        )}

        {/* 当前模式说明 */}
        <div className="p-3 bg-blue-50 rounded-lg">
          <div className="flex items-center space-x-2 mb-1">
            {getModeIcon(mode)}
            <span className="text-sm font-medium text-blue-900">当前模式</span>
          </div>
          <p className="text-sm text-blue-700">{getModeDescription(mode)}</p>
        </div>

        {/* 转盘/列表切换 */}
        <div className="flex gap-2">
          <Button
            variant={!showWheel ? "default" : "outline"}
            onClick={() => setShowWheel(false)}
            className="flex-1"
          >
            <Play className="h-4 w-4 mr-2" />
            列表模式
          </Button>
          <Button
            variant={showWheel ? "default" : "outline"}
            onClick={() => setShowWheel(true)}
            className="flex-1"
          >
            <Target className="h-4 w-4 mr-2" />
            转盘模式
          </Button>
        </div>

        {/* 转盘显示 */}
        {showWheel && (
          <div className="space-y-4">
            <div className="relative bg-gradient-to-br from-blue-50 to-indigo-50 rounded-lg p-4 border-2 border-blue-200" style={{ height: '300px' }}>
              <canvas
                ref={canvasRef}
                className="w-full h-full"
                style={{ maxWidth: '100%', maxHeight: '100%' }}
              />
              {selectedStudent && !isAnimating && (
                <div className="absolute top-2 right-2 bg-green-500 text-white px-3 py-1 rounded-full text-sm font-semibold animate-bounce">
                  🎉 {selectedStudent.name}
                </div>
              )}
            </div>
            
            <motion.div
              whileHover={{ scale: 1.02 }}
              whileTap={{ scale: 0.98 }}
            >
              <Button
                onClick={spinWheel}
                disabled={students.length === 0 || isAnimating}
                className="w-full h-16 text-lg font-semibold bg-gradient-to-r from-red-500 to-pink-600 hover:from-red-600 hover:to-pink-700"
              >
                {isAnimating ? (
                  <div className="flex items-center space-x-2">
                    <RotateCw className="h-6 w-6 animate-spin" />
                    <span>转盘旋转中...</span>
                  </div>
                ) : (
                  <div className="flex items-center space-x-2">
                    <Target className="h-6 w-6" />
                    <span>开始转盘</span>
                  </div>
                )}
              </Button>
            </motion.div>
          </div>
        )}

        {/* 列表模式按钮 */}
        {!showWheel && (
          <motion.div
            whileHover={{ scale: 1.02 }}
            whileTap={{ scale: 0.98 }}
          >
            <Button
              onClick={handlePick}
              disabled={students.length === 0 || isAnimating}
              className="w-full h-16 text-lg font-semibold bg-gradient-to-r from-blue-500 to-indigo-600 hover:from-blue-600 hover:to-indigo-700"
            >
              {isAnimating ? (
                <div className="flex items-center space-x-2">
                  <Shuffle className="h-6 w-6" />
                  <span>点名中...</span>
                </div>
              ) : (
                <div className="flex items-center space-x-2">
                  <Play className="h-6 w-6" />
                  <span>开始点名</span>
                </div>
              )}
            </Button>
          </motion.div>
        )}

        {/* 统计信息 */}
        <div className="grid grid-cols-2 gap-4 pt-4 border-t">
          <div className="text-center">
            <div className="text-2xl font-bold text-blue-600">{students.length}</div>
            <div className="text-sm text-gray-600">总人数</div>
          </div>
          <div className="text-center">
            <div className="text-2xl font-bold text-green-600">
              {mode === 'single' ? 1 : 
               mode === 'multiple' ? Math.min(multipleCount, students.length) :
               groupCount}
            </div>
            <div className="text-sm text-gray-600">
              {mode === 'group' ? '分组数' : '选择数'}
            </div>
          </div>
        </div>
      </CardContent>
    </Card>
  );
}