import React, { useCallback, useEffect } from 'react';
import { motion, AnimatePresence } from 'framer-motion';
import { useDropzone } from 'react-dropzone';
import { Card, CardContent, CardHeader, CardTitle } from './ui/card';
import { Button } from './ui/button';
import { Badge } from './ui/badge';
import { useToast } from './ui/use-toast';
import { Upload, X, Trash2, Users, Save, FolderOpen } from 'lucide-react';
import { StorageManager, STORAGE_KEYS } from '../utils/storage';
import type { Student } from '../App';

interface NameListManagerProps {
  students: Student[];
  onImport: (students: Student[]) => void;
  onRemoveStudent: (studentId: string) => void;
  onClearStudents: () => void;
}

export function NameListManager({
  students,
  onImport,
  onRemoveStudent,
  onClearStudents
}: NameListManagerProps) {
  const { toast } = useToast();

  // 保存导入历史
  const saveImportHistory = (students: Student[], filename: string) => {
    const importHistory = StorageManager.get('import-history', []);
    const newEntry = {
      id: Date.now(),
      filename,
      studentCount: students.length,
      timestamp: new Date().toISOString(),
      students: students.map(s => s.name)
    };
    
    const updatedHistory = [newEntry, ...importHistory].slice(0, 10); // 保留最近10次导入
    StorageManager.set('import-history', updatedHistory);
  };

  // 快速恢复上次导入的名单
  const restoreLastImport = () => {
    const importHistory = StorageManager.get('import-history', []);
    if (importHistory.length > 0) {
      const lastImport = importHistory[0];
      const restoredStudents: Student[] = lastImport.students.map((name: string, index: number) => ({
        id: `restored-${Date.now()}-${index}`,
        name
      }));
      
      onImport(restoredStudents);
      toast({
        title: "名单已恢复",
        description: `已恢复 "${lastImport.filename}" 中的 ${lastImport.studentCount} 个学生`,
      });
    } else {
      toast({
        title: "无历史记录",
        description: "没有找到之前导入的名单记录",
        variant: "destructive",
      });
    }
  };

  const onDrop = useCallback((acceptedFiles: File[]) => {
    const file = acceptedFiles[0];
    if (!file) return;

    if (!file.name.endsWith('.txt')) {
      toast({
        title: "文件格式错误",
        description: "请选择 .txt 格式的文件",
        variant: "destructive",
      });
      return;
    }

    const reader = new FileReader();
    reader.onload = (e) => {
      const content = e.target?.result as string;
      const names = content
        .split('\n')
        .map(name => name.trim())
        .filter(name => name.length > 0);

      if (names.length === 0) {
        toast({
          title: "文件内容为空",
          description: "请确保文件中包含学生姓名",
          variant: "destructive",
        });
        return;
      }

      const newStudents: Student[] = names.map((name, index) => ({
        id: `student-${Date.now()}-${index}`,
        name
      }));

      // 保存导入历史
      saveImportHistory(newStudents, file.name);

      onImport(newStudents);
      toast({
        title: "导入成功",
        description: `成功导入 ${names.length} 个学生姓名`,
      });
    };

    reader.readAsText(file, 'UTF-8');
  }, [onImport, toast]);

  const { getRootProps, getInputProps, isDragActive } = useDropzone({
    onDrop,
    accept: {
      'text/plain': ['.txt']
    },
    multiple: false
  });

  return (
    <Card className="h-full">
      <CardHeader>
        <CardTitle className="flex items-center space-x-2">
          <Users className="h-5 w-5" />
          <span>名单管理</span>
          <Badge variant="secondary">{students.length}</Badge>
        </CardTitle>
      </CardHeader>
      
      <CardContent className="space-y-4">
        {/* 文件上传区域 */}
        <div
          {...getRootProps()}
          className={`
            border-2 border-dashed rounded-lg p-6 text-center cursor-pointer transition-colors
            ${isDragActive 
              ? 'border-blue-500 bg-blue-50' 
              : 'border-gray-300 hover:border-blue-400 hover:bg-gray-50'
            }
          `}
        >
          <input {...getInputProps()} />
          <Upload className="h-8 w-8 mx-auto mb-2 text-gray-400" />
          <p className="text-sm text-gray-600">
            {isDragActive
              ? '松开鼠标导入文件'
              : '拖拽 TXT 文件到此处，或点击选择文件'
            }
          </p>
          <p className="text-xs text-gray-500 mt-1">
            支持 .txt 格式，每行一个姓名
          </p>
        </div>

        {/* 学生名单显示 */}
        <div className="space-y-2">
          <div className="flex items-center justify-between">
            <h3 className="text-sm font-medium text-gray-700">当前名单</h3>
            <div className="flex space-x-2">
              <Button
                variant="outline"
                size="sm"
                onClick={restoreLastImport}
                className="text-blue-600 hover:text-blue-700"
              >
                <FolderOpen className="h-4 w-4 mr-1" />
                恢复
              </Button>
              {students.length > 0 && (
                <Button
                  variant="outline"
                  size="sm"
                  onClick={onClearStudents}
                  className="text-red-600 hover:text-red-700"
                >
                  <Trash2 className="h-4 w-4 mr-1" />
                  清空
                </Button>
              )}
            </div>
          </div>

          <div className="max-h-64 overflow-y-auto space-y-1">
            <AnimatePresence>
              {students.map((student) => (
                <motion.div
                  key={student.id}
                  initial={{ opacity: 0, x: -20 }}
                  animate={{ opacity: 1, x: 0 }}
                  exit={{ opacity: 0, x: 20 }}
                  transition={{ duration: 0.2 }}
                  className="flex items-center justify-between p-2 bg-gray-50 rounded-md"
                >
                  <span className="text-sm text-gray-700">{student.name}</span>
                  <Button
                    variant="ghost"
                    size="sm"
                    onClick={() => onRemoveStudent(student.id)}
                    className="h-6 w-6 p-0 text-gray-400 hover:text-red-500"
                  >
                    <X className="h-4 w-4" />
                  </Button>
                </motion.div>
              ))}
            </AnimatePresence>
          </div>

          {students.length === 0 && (
            <div className="text-center py-8 text-gray-500">
              <Users className="h-12 w-12 mx-auto mb-2 text-gray-300" />
              <p className="text-sm">暂无学生名单</p>
              <p className="text-xs">请导入 TXT 文件添加学生</p>
            </div>
          )}
        </div>
      </CardContent>
    </Card>
  );
}