import React from 'react';
import { motion } from 'framer-motion';
import { Dashboard } from './dashboard';
import { NameListManager } from './name-list-manager';
import { PickerControl } from './picker-control';
import { ResultDisplay } from './result-display';
import { ClassroomTools } from './classroom-tools';
import Schedule from './schedule';
import { ClassroomVoting } from './classroom-voting';
import { AdvancedTools } from './advanced-tools';
import { Attendance } from './attendance';
import { SettingsPanel } from './settings-panel';
import { Student, PickResult, Settings } from '../App';

interface MainContentProps {
  activeSection: string;
  students: Student[];
  currentResult: PickResult | null;
  history: PickResult[];
  settings: Settings;
  onStudentsImport: (students: Student[]) => void;
  onPickResult: (result: PickResult) => void;
  onRemoveStudent: (studentId: string) => void;
  onClearStudents: () => void;
  onSettingsChange: (settings: Settings) => void;
  onSectionChange: (section: string) => void;
}

export function MainContent({
  activeSection,
  students,
  currentResult,
  history,
  settings,
  onStudentsImport,
  onPickResult,
  onRemoveStudent,
  onClearStudents,
  onSettingsChange,
  onSectionChange
}: MainContentProps) {
  const renderContent = () => {
    switch (activeSection) {
      case 'dashboard':
        return (
          <Dashboard
            students={students}
            currentResult={currentResult}
            history={history}
            onPickResult={onPickResult}
            onSectionChange={onSectionChange}
          />
        );

      case 'students':
        return (
          <div className="max-w-6xl mx-auto">
            <NameListManager
              students={students}
              onImport={onStudentsImport}
              onRemoveStudent={onRemoveStudent}
              onClearStudents={onClearStudents}
            />
          </div>
        );

      case 'picker':
        return (
          <div className="grid grid-cols-1 xl:grid-cols-3 gap-6 max-w-7xl mx-auto">
            <div className="xl:col-span-1">
              <NameListManager
                students={students}
                onImport={onStudentsImport}
                onRemoveStudent={onRemoveStudent}
                onClearStudents={onClearStudents}
              />
            </div>
            <div className="xl:col-span-1">
              <PickerControl
                students={students}
                onPickResult={onPickResult}
                settings={settings}
              />
            </div>
            <div className="xl:col-span-1">
              <ResultDisplay
                currentResult={currentResult}
                history={history}
                settings={settings}
              />
            </div>
          </div>
        );

      case 'classroom':
        return (
          <div className="max-w-6xl mx-auto">
            <ClassroomTools
              students={students}
              onPickResult={onPickResult}
            />
          </div>
        );

      case 'schedule':
        return (
          <div className="max-w-7xl mx-auto">
            <Schedule />
          </div>
        );

      case 'voting':
        return (
          <div className="max-w-6xl mx-auto">
            <ClassroomVoting students={students} />
          </div>
        );

      case 'attendance':
        return (
          <div className="max-w-6xl mx-auto">
            <Attendance students={students} />
          </div>
        );

      case 'advanced':
        return (
          <div className="max-w-6xl mx-auto">
            <AdvancedTools
              students={students}
              history={history}
              onPickResult={onPickResult}
            />
          </div>
        );

      case 'settings':
        return (
          <div className="max-w-4xl mx-auto">
            <SettingsPanel
              settings={settings}
              onSettingsChange={onSettingsChange}
              students={students}
              history={history}
            />
          </div>
        );

      default:
        return (
          <Dashboard
            students={students}
            currentResult={currentResult}
            history={history}
            onPickResult={onPickResult}
            onSectionChange={onSectionChange}
          />
        );
    }
  };

  return (
    <motion.div
      initial={{ opacity: 0 }}
      animate={{ opacity: 1 }}
      transition={{ duration: 0.3 }}
      className="min-h-screen"
    >
      {renderContent()}
    </motion.div>
  );
}