import React from 'react';

interface ErrorBoundaryState {
  hasError: boolean;
  error?: Error;
}

interface ErrorBoundaryProps {
  children: React.ReactNode;
  fallback?: React.ReactNode;
}

export class ErrorBoundary extends React.Component<ErrorBoundaryProps, ErrorBoundaryState> {
  constructor(props: ErrorBoundaryProps) {
    super(props);
    this.state = { hasError: false };
  }

  static getDerivedStateFromError(error: Error): ErrorBoundaryState {
    return { hasError: true, error };
  }

  componentDidCatch(error: Error, errorInfo: React.ErrorInfo) {
    // 忽略 getBoundingClientRect 相关的错误
    if (error.message.includes('getBoundingClientRect') || 
        error.message.includes('Cannot read properties of null')) {
      console.warn('忽略 DOM 访问错误:', error.message);
      this.setState({ hasError: false });
      return;
    }
    
    console.error('ErrorBoundary 捕获到错误:', error, errorInfo);
  }

  render() {
    if (this.state.hasError) {
      return this.props.fallback || (
        <div className="p-4 bg-red-50 border border-red-200 rounded-lg">
          <h3 className="text-red-800 font-medium">出现了一个错误</h3>
          <p className="text-red-600 text-sm mt-1">页面遇到问题，请刷新重试</p>
        </div>
      );
    }

    return this.props.children;
  }
}