import React, { useState, useEffect } from 'react';
import { motion } from 'framer-motion';
import { Card, CardContent, CardHeader, CardTitle } from './ui/card';
import { Badge } from './ui/badge';
import { Button } from './ui/button';
import { Progress } from './ui/progress';
import { 
  Users, 
  Clock, 
  TrendingUp, 
  Calendar,
  Shuffle,
  BarChart3,
  Star,
  Award,
  Target,
  Zap,
  BookOpen,
  Activity,
  UserCheck,
  UserX,
  AlertCircle,
  Dices,
  ArrowRight
} from 'lucide-react';
import { Student, PickResult } from '../App';

interface AttendanceRecord {
  id: string;
  studentId: string;
  date: string;
  status: 'present' | 'absent' | 'late' | 'excused';
  timestamp: Date;
  note?: string;
}

interface DashboardProps {
  students: Student[];
  currentResult: PickResult | null;
  history: PickResult[];
  onPickResult: (result: PickResult) => void;
  onSectionChange?: (section: string) => void;
}

export function Dashboard({ students, currentResult, history, onPickResult, onSectionChange }: DashboardProps) {
  const [attendanceRecords, setAttendanceRecords] = useState<AttendanceRecord[]>([]);

  // 从localStorage加载考勤记录
  useEffect(() => {
    const savedRecords = localStorage.getItem('attendance-records');
    if (savedRecords) {
      try {
        const parsedRecords = JSON.parse(savedRecords);
        const recordsWithDates = parsedRecords.map((record: any) => ({
          ...record,
          timestamp: new Date(record.timestamp)
        }));
        setAttendanceRecords(recordsWithDates);
      } catch (error) {
        console.error('Failed to load attendance records:', error);
      }
    }
  }, []);

  // 统计数据
  const totalStudents = students.length;
  const todayPicks = history.filter(h => {
    const today = new Date();
    const pickDate = new Date(h.timestamp);
    return pickDate.toDateString() === today.toDateString();
  }).length;

  const weeklyPicks = history.filter(h => {
    const weekAgo = new Date();
    weekAgo.setDate(weekAgo.getDate() - 7);
    return new Date(h.timestamp) > weekAgo;
  }).length;

  // 计算今日考勤统计
  const getTodayAttendanceStats = () => {
    const today = new Date().toISOString().split('T')[0];
    const todayRecords = attendanceRecords.filter(record => record.date === today);
    
    const stats = {
      present: 0,
      absent: 0,
      late: 0,
      excused: 0,
      unmarked: totalStudents
    };

    todayRecords.forEach(record => {
      stats[record.status]++;
      stats.unmarked--;
    });

    const attendanceRate = totalStudents > 0 ? ((stats.present + stats.late) / totalStudents) * 100 : 0;
    
    return { ...stats, attendanceRate };
  };

  const attendanceStats = getTodayAttendanceStats();

  // 快速点名功能
  const handleQuickPick = () => {
    if (students.length === 0) return;
    
    const randomStudent = students[Math.floor(Math.random() * students.length)];
    const result: PickResult = {
      students: [randomStudent],
      timestamp: new Date(),
      mode: 'single',
      context: '仪表盘快速点名'
    };
    onPickResult(result);
  };

  return (
    <div className="space-y-6">
      {/* 欢迎横幅 */}
      <motion.div
        initial={{ opacity: 0, y: -20 }}
        animate={{ opacity: 1, y: 0 }}
        transition={{ duration: 0.5 }}
        className="bg-gradient-to-r from-blue-500 via-purple-500 to-pink-500 rounded-2xl p-8 text-white relative overflow-hidden"
      >
        <div className="relative z-10">
          <h1 className="text-3xl font-bold mb-2">欢迎使用柯猿课堂互动系统</h1>
          <p className="text-blue-100 text-lg mb-6">
            今天是 {new Date().toLocaleDateString('zh-CN', { 
              year: 'numeric', 
              month: 'long', 
              day: 'numeric',
              weekday: 'long'
            })}
          </p>
          <div className="flex items-center space-x-4">
            <Button 
              onClick={handleQuickPick}
              disabled={students.length === 0}
              className="bg-white/20 hover:bg-white/30 text-white border-white/30"
            >
              <Dices className="w-4 h-4 mr-2" />
              快速点名
            </Button>
            <Button 
              variant="outline" 
              className="bg-white/10 hover:bg-white/20 text-white border-white/30"
              onClick={() => onSectionChange?.('schedule')}
            >
              <Calendar className="w-4 h-4 mr-2" />
              查看课程表
            </Button>
          </div>
        </div>
        <div className="absolute top-0 right-0 w-64 h-64 bg-white/10 rounded-full -translate-y-32 translate-x-32"></div>
        <div className="absolute bottom-0 left-0 w-48 h-48 bg-white/5 rounded-full translate-y-24 -translate-x-24"></div>
      </motion.div>

      {/* 统计卡片 */}
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
        <motion.div
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.5, delay: 0.1 }}
        >
          <Card className="bg-gradient-to-br from-blue-50 to-blue-100 border-blue-200">
            <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
              <CardTitle className="text-sm font-medium text-blue-700">学生总数</CardTitle>
              <Users className="h-4 w-4 text-blue-600" />
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold text-blue-900">{totalStudents}</div>
              <p className="text-xs text-blue-600 mt-1">
                活跃学生 {totalStudents} 名
              </p>
            </CardContent>
          </Card>
        </motion.div>

        <motion.div
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.5, delay: 0.2 }}
        >
          <Card className="bg-gradient-to-br from-green-50 to-green-100 border-green-200">
            <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
              <CardTitle className="text-sm font-medium text-green-700">今日出勤</CardTitle>
              <UserCheck className="h-4 w-4 text-green-600" />
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold text-green-900">{attendanceStats.present}</div>
              <p className="text-xs text-green-600 mt-1">
                出勤率 {attendanceStats.attendanceRate.toFixed(1)}%
              </p>
            </CardContent>
          </Card>
        </motion.div>

        <motion.div
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.5, delay: 0.3 }}
        >
          <Card className="bg-gradient-to-br from-red-50 to-red-100 border-red-200">
            <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
              <CardTitle className="text-sm font-medium text-red-700">今日缺席</CardTitle>
              <UserX className="h-4 w-4 text-red-600" />
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold text-red-900">{attendanceStats.absent}</div>
              <p className="text-xs text-red-600 mt-1">
                迟到 {attendanceStats.late} 人
              </p>
            </CardContent>
          </Card>
        </motion.div>

        <motion.div
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.5, delay: 0.4 }}
        >
          <Card className="bg-gradient-to-br from-purple-50 to-purple-100 border-purple-200">
            <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
              <CardTitle className="text-sm font-medium text-purple-700">本周点名</CardTitle>
              <TrendingUp className="h-4 w-4 text-purple-600" />
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold text-purple-900">{weeklyPicks}</div>
              <p className="text-xs text-purple-600 mt-1">
                今日 {todayPicks} 次
              </p>
            </CardContent>
          </Card>
        </motion.div>
      </div>

      {/* 考勤概览 */}
      {totalStudents > 0 && (
        <motion.div
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.5, delay: 0.5 }}
        >
          <Card className="bg-gradient-to-r from-teal-50 to-cyan-50 border-teal-200">
            <CardHeader>
              <CardTitle className="flex items-center justify-between">
                <div className="flex items-center gap-2">
                  <UserCheck className="w-5 h-5 text-teal-600" />
                  今日考勤概览
                </div>
                <Badge variant="outline" className="text-teal-700 border-teal-300">
                  {new Date().toLocaleDateString('zh-CN')}
                </Badge>
              </CardTitle>
            </CardHeader>
            <CardContent>
              <div className="space-y-4">
                {/* 出勤率进度条 */}
                <div>
                  <div className="flex items-center justify-between mb-2">
                    <span className="text-sm font-medium text-gray-700">出勤率</span>
                    <span className="text-sm text-gray-600">{attendanceStats.attendanceRate.toFixed(1)}%</span>
                  </div>
                  <Progress value={attendanceStats.attendanceRate} className="h-3" />
                </div>

                {/* 考勤统计 */}
                <div className="grid grid-cols-2 md:grid-cols-4 gap-4">
                  <div className="text-center p-3 bg-green-50 rounded-lg">
                    <div className="text-lg font-bold text-green-700">{attendanceStats.present}</div>
                    <div className="text-xs text-green-600">出席</div>
                  </div>
                  <div className="text-center p-3 bg-yellow-50 rounded-lg">
                    <div className="text-lg font-bold text-yellow-700">{attendanceStats.late}</div>
                    <div className="text-xs text-yellow-600">迟到</div>
                  </div>
                  <div className="text-center p-3 bg-red-50 rounded-lg">
                    <div className="text-lg font-bold text-red-700">{attendanceStats.absent}</div>
                    <div className="text-xs text-red-600">缺席</div>
                  </div>
                  <div className="text-center p-3 bg-blue-50 rounded-lg">
                    <div className="text-lg font-bold text-blue-700">{attendanceStats.excused}</div>
                    <div className="text-xs text-blue-600">请假</div>
                  </div>
                </div>

                {attendanceStats.unmarked > 0 && (
                  <div className="flex items-center justify-between p-3 bg-gray-50 rounded-lg">
                    <div className="flex items-center gap-2">
                      <AlertCircle className="w-4 h-4 text-gray-500" />
                      <span className="text-sm text-gray-700">还有 {attendanceStats.unmarked} 名学生未标记考勤</span>
                    </div>
                    <Button 
                      size="sm" 
                      variant="outline"
                      onClick={() => onSectionChange?.('attendance')}
                    >
                      去标记
                    </Button>
                  </div>
                )}
              </div>
            </CardContent>
          </Card>
        </motion.div>
      )}

      {/* 主要功能区域 */}
      <div className="grid grid-cols-1 lg:grid-cols-3 gap-6">
        {/* 最近点名结果 */}
        <motion.div
          initial={{ opacity: 0, x: -20 }}
          animate={{ opacity: 1, x: 0 }}
          transition={{ duration: 0.5, delay: 0.6 }}
          className="lg:col-span-2"
        >
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center">
                <Clock className="w-5 h-5 mr-2 text-blue-600" />
                最近点名记录
              </CardTitle>
            </CardHeader>
            <CardContent>
              {history.length > 0 ? (
                <div className="space-y-4">
                  {history.slice(0, 5).map((record, index) => (
                    <motion.div
                      key={index}
                      initial={{ opacity: 0, y: 10 }}
                      animate={{ opacity: 1, y: 0 }}
                      transition={{ delay: index * 0.1 }}
                      className="flex items-center justify-between p-4 bg-gray-50 rounded-lg hover:bg-gray-100 transition-colors"
                    >
                      <div className="flex items-center space-x-3">
                        <div className="w-10 h-10 bg-gradient-to-br from-blue-500 to-purple-600 rounded-full flex items-center justify-center">
                          <span className="text-white font-medium text-sm">
                            {record.students[0]?.name?.charAt(0) || '?'}
                          </span>
                        </div>
                        <div>
                          <p className="font-medium text-gray-900">
                            {record.students.map(s => s.name).join(', ')}
                          </p>
                          <p className="text-sm text-gray-500">
                            {new Date(record.timestamp).toLocaleString('zh-CN')}
                          </p>
                        </div>
                      </div>
                      <Badge variant={record.mode === 'single' ? 'default' : 'secondary'}>
                        {record.mode === 'single' ? '单人' : record.mode === 'multiple' ? '多人' : '分组'}
                      </Badge>
                    </motion.div>
                  ))}
                </div>
              ) : (
                <div className="text-center py-8">
                  <Dices className="w-12 h-12 text-gray-400 mx-auto mb-4" />
                  <p className="text-gray-500">还没有点名记录</p>
                  <p className="text-sm text-gray-400 mt-1">开始你的第一次点名吧！</p>
                </div>
              )}
            </CardContent>
          </Card>
        </motion.div>

        {/* 快速操作 */}
        <motion.div
          initial={{ opacity: 0, x: 20 }}
          animate={{ opacity: 1, x: 0 }}
          transition={{ duration: 0.5, delay: 0.7 }}
        >
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center">
                <Zap className="w-5 h-5 mr-2 text-yellow-600" />
                快速操作
              </CardTitle>
            </CardHeader>
            <CardContent className="space-y-3">
              <Button 
                onClick={handleQuickPick}
                disabled={students.length === 0}
                className="w-full justify-between bg-gradient-to-r from-blue-500 to-purple-600 hover:from-blue-600 hover:to-purple-700"
              >
                <span className="flex items-center">
                  <Dices className="w-4 h-4 mr-2" />
                  课堂互动
                </span>
                <ArrowRight className="w-4 h-4" />
              </Button>
              
              <Button 
                variant="outline" 
                className="w-full justify-between"
                onClick={() => onSectionChange?.('attendance')}
              >
                <span className="flex items-center">
                  <UserCheck className="w-4 h-4 mr-2" />
                  考勤管理
                </span>
                <ArrowRight className="w-4 h-4" />
              </Button>
              
              <Button 
                variant="outline" 
                className="w-full justify-between"
                onClick={() => onSectionChange?.('students')}
              >
                <span className="flex items-center">
                  <Users className="w-4 h-4 mr-2" />
                  管理学生
                </span>
                <ArrowRight className="w-4 h-4" />
              </Button>
              
              <Button 
                variant="outline" 
                className="w-full justify-between"
                onClick={() => onSectionChange?.('advanced')}
              >
                <span className="flex items-center">
                  <BarChart3 className="w-4 h-4 mr-2" />
                  查看统计
                </span>
                <ArrowRight className="w-4 h-4" />
              </Button>
            </CardContent>
          </Card>
        </motion.div>
      </div>

      {/* 当前点名结果 */}
      {currentResult && (
        <motion.div
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.5 }}
        >
          <Card className="bg-gradient-to-r from-green-50 to-blue-50 border-green-200">
            <CardHeader>
              <CardTitle className="flex items-center text-green-700">
                <Award className="w-5 h-5 mr-2" />
                最新点名结果
              </CardTitle>
            </CardHeader>
            <CardContent>
              <div className="flex items-center justify-between">
                <div className="flex items-center space-x-4">
                  <div className="w-16 h-16 bg-gradient-to-br from-green-500 to-blue-600 rounded-full flex items-center justify-center">
                    <span className="text-white font-bold text-xl">
                      {currentResult.students[0]?.name?.charAt(0) || '?'}
                    </span>
                  </div>
                  <div>
                    <h3 className="text-xl font-bold text-gray-900">
                      {currentResult.students.map(s => s.name).join(', ')}
                    </h3>
                    <p className="text-gray-600">
                      {new Date(currentResult.timestamp).toLocaleString('zh-CN')}
                    </p>
                  </div>
                </div>
                <Badge className="bg-green-100 text-green-700">
                  <Star className="w-3 h-3 mr-1" />
                  最新
                </Badge>
              </div>
            </CardContent>
          </Card>
        </motion.div>
      )}
    </div>
  );
}